#!/bin/bash

###############################################################################
# OllamaDeploy - Vollautomatische Installation
# Einzeiler-Installation eines lokalen KI-Systems auf macOS
#
# Nutzung:
#   ./install.sh            # Interaktiver Modus
#   ./install.sh --auto     # Vollautomatisch
#   ./install.sh --help     # Hilfe anzeigen
###############################################################################

set -e  # Beende bei Fehler
set -o pipefail

# ============================================================================
# KONFIGURATION
# ============================================================================

VERSION="1.0.0"
AUTO_MODE=false
VERBOSE=false
SKIP_MODEL_DOWNLOAD=false

# Docker Desktop Download URL (neueste Version)
DOCKER_DMG_URL="https://desktop.docker.com/mac/main/arm64/Docker.dmg"
DOCKER_INTEL_DMG_URL="https://desktop.docker.com/mac/main/amd64/Docker.dmg"

# Default Model
DEFAULT_MODEL="llama3.1"

# Temporäres Verzeichnis
TMP_DIR="/tmp/ollamadeploy-install"

# ============================================================================
# FARBEN UND SYMBOLE
# ============================================================================

if [[ -t 1 ]]; then
    RED='\033[0;31m'
    GREEN='\033[0;32m'
    YELLOW='\033[1;33m'
    BLUE='\033[0;34m'
    MAGENTA='\033[0;35m'
    CYAN='\033[0;36m'
    BOLD='\033[1m'
    DIM='\033[2m'
    NC='\033[0m'
else
    RED=''
    GREEN=''
    YELLOW=''
    BLUE=''
    MAGENTA=''
    CYAN=''
    BOLD=''
    DIM=''
    NC=''
fi

CHECKMARK="✓"
CROSS="✗"
ARROW="→"
GEAR="⚙"
DOWNLOAD="⬇"
ROCKET="🚀"
SPARKLE="✨"

# ============================================================================
# LOGGING FUNKTIONEN
# ============================================================================

print_header() {
    clear
    echo -e "${BLUE}${BOLD}"
    echo "╔════════════════════════════════════════════════════════════════╗"
    echo "║                                                                ║"
    echo "║        ${ROCKET}  OllamaDeploy Installer v${VERSION}  ${ROCKET}               ║"
    echo "║                                                                ║"
    echo "║            Lokales KI-System für macOS                         ║"
    echo "║                                                                ║"
    echo "╚════════════════════════════════════════════════════════════════╝"
    echo -e "${NC}\n"
}

log_info() {
    echo -e "${BLUE}[ℹ]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[${CHECKMARK}]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[⚠]${NC} $1"
}

log_error() {
    echo -e "${RED}[${CROSS}]${NC} $1"
}

log_step() {
    echo -e "\n${CYAN}${BOLD}${ARROW} $1${NC}"
}

log_substep() {
    echo -e "  ${DIM}${GEAR}${NC} $1"
}

print_progress() {
    local current=$1
    local total=$2
    local text=$3
    local percent=$((current * 100 / total))
    local filled=$((percent / 2))
    local empty=$((50 - filled))

    printf "\r${BLUE}["
    printf "%${filled}s" | tr ' ' '█'
    printf "%${empty}s" | tr ' ' '░'
    printf "]${NC} ${percent}%% ${text}"
}

finish_progress() {
    echo ""
}

# ============================================================================
# HILFSFUNKTIONEN
# ============================================================================

show_help() {
    cat << EOF
${BOLD}OllamaDeploy Installer v${VERSION}${NC}

Vollautomatische Installation eines lokalen KI-Systems auf macOS.

${BOLD}VERWENDUNG:${NC}
    ./install.sh [OPTIONEN]

${BOLD}OPTIONEN:${NC}
    --auto              Vollautomatischer Modus (keine Rückfragen)
    --skip-model        Modell-Download überspringen
    --verbose           Ausführliche Ausgabe
    --help              Diese Hilfe anzeigen

${BOLD}BEISPIELE:${NC}
    ./install.sh                    # Interaktive Installation
    ./install.sh --auto             # Vollautomatisch
    ./install.sh --auto --skip-model # Ohne Modell-Download

${BOLD}WAS WIRD INSTALLIERT:${NC}
    • Homebrew (Package Manager)
    • Ollama (LLM Runtime)
    • Docker Desktop
    • Open WebUI (Web-Interface)
    • Optional: KI-Modell (llama3.1)

${BOLD}SYSTEMANFORDERUNGEN:${NC}
    • macOS 11 Big Sur oder neuer
    • Apple Silicon (M1/M2/M3/M4) empfohlen
    • 16 GB RAM (32+ GB empfohlen)
    • 50 GB freier Speicherplatz

${BOLD}WEITERE INFO:${NC}
    Repository: https://github.com/cocoacastsplus/OllamaDeploy
    Dokumentation: docs/GETTING_STARTED.md
EOF
    exit 0
}

parse_arguments() {
    while [[ $# -gt 0 ]]; do
        case $1 in
            --auto)
                AUTO_MODE=true
                shift
                ;;
            --skip-model)
                SKIP_MODEL_DOWNLOAD=true
                shift
                ;;
            --verbose)
                VERBOSE=true
                set -x
                shift
                ;;
            --help|-h)
                show_help
                ;;
            *)
                log_error "Unbekannte Option: $1"
                echo "Nutze --help für weitere Informationen"
                exit 1
                ;;
        esac
    done
}

ask_confirmation() {
    if [[ "$AUTO_MODE" == "true" ]]; then
        return 0
    fi

    local prompt="$1"
    local default="${2:-y}"

    if [[ "$default" == "y" ]]; then
        prompt="$prompt [Y/n] "
    else
        prompt="$prompt [y/N] "
    fi

    read -p "$prompt" -n 1 -r
    echo

    if [[ "$default" == "y" ]]; then
        [[ $REPLY =~ ^[Yy]$ ]] || [[ -z $REPLY ]]
    else
        [[ $REPLY =~ ^[Yy]$ ]]
    fi
}

# ============================================================================
# SYSTEM-CHECKS
# ============================================================================

check_macos() {
    log_step "Prüfe Betriebssystem..."

    if [[ "$OSTYPE" != "darwin"* ]]; then
        log_error "Dieses Script läuft nur auf macOS!"
        exit 1
    fi

    local macos_version=$(sw_vers -productVersion)
    local major_version=$(echo "$macos_version" | cut -d '.' -f 1)

    log_success "macOS $macos_version erkannt"

    if [[ $major_version -lt 11 ]]; then
        log_warning "macOS 11 Big Sur oder neuer wird empfohlen"
        if ! ask_confirmation "Trotzdem fortfahren?" "n"; then
            exit 1
        fi
    fi
}

check_architecture() {
    log_substep "Prüfe CPU-Architektur..."

    local arch=$(uname -m)

    if [[ "$arch" == "arm64" ]]; then
        log_success "Apple Silicon (M1/M2/M3/M4) erkannt - Optimal!"
        DOCKER_DMG_URL="$DOCKER_DMG_URL"
    else
        log_warning "Intel-Architektur erkannt - Performance eingeschränkt"
        DOCKER_DMG_URL="$DOCKER_INTEL_DMG_URL"

        if [[ "$AUTO_MODE" != "true" ]]; then
            log_info "Apple Silicon wird für beste Performance empfohlen"
            if ! ask_confirmation "Trotzdem fortfahren?" "y"; then
                exit 1
            fi
        fi
    fi
}

check_memory() {
    log_substep "Prüfe Arbeitsspeicher..."

    local total_mem=$(sysctl -n hw.memsize)
    local total_gb=$((total_mem / 1024 / 1024 / 1024))

    log_info "Installierter RAM: ${total_gb} GB"

    if [[ $total_gb -lt 16 ]]; then
        log_warning "Weniger als 16 GB RAM - Nur kleine Modelle nutzbar"
        if [[ "$AUTO_MODE" != "true" ]]; then
            if ! ask_confirmation "Trotzdem fortfahren?" "n"; then
                exit 1
            fi
        fi
    elif [[ $total_gb -ge 32 ]]; then
        log_success "32+ GB RAM - Optimal für große Modelle!"
    else
        log_success "16 GB RAM - Gut für Standard-Modelle"
    fi
}

check_disk_space() {
    log_substep "Prüfe Speicherplatz..."

    local free_space=$(df -g / | tail -1 | awk '{print $4}')

    log_info "Freier Speicher: ${free_space} GB"

    if [[ $free_space -lt 50 ]]; then
        log_warning "Weniger als 50 GB frei - Speicherplatz könnte knapp werden"
        if [[ "$AUTO_MODE" != "true" ]]; then
            if ! ask_confirmation "Trotzdem fortfahren?" "n"; then
                exit 1
            fi
        fi
    else
        log_success "Ausreichend Speicherplatz verfügbar"
    fi
}

check_internet() {
    log_substep "Prüfe Internetverbindung..."

    if ping -c 1 -W 2 8.8.8.8 &> /dev/null; then
        log_success "Internetverbindung verfügbar"
    else
        log_error "Keine Internetverbindung!"
        log_info "Bitte stelle eine Verbindung her und versuche es erneut"
        exit 1
    fi
}

# ============================================================================
# HOMEBREW INSTALLATION
# ============================================================================

install_homebrew() {
    log_step "Installiere Homebrew..."

    if command -v brew &> /dev/null; then
        local brew_version=$(brew --version | head -n 1)
        log_success "Homebrew bereits installiert: $brew_version"
        return 0
    fi

    log_info "Homebrew wird installiert..."
    log_info "Dies kann einige Minuten dauern..."

    # Homebrew Installation (non-interactive)
    NONINTERACTIVE=1 /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"

    # Homebrew zum PATH hinzufügen
    if [[ -f "/opt/homebrew/bin/brew" ]]; then
        eval "$(/opt/homebrew/bin/brew shellenv)"
        echo 'eval "$(/opt/homebrew/bin/brew shellenv)"' >> ~/.zprofile
    elif [[ -f "/usr/local/bin/brew" ]]; then
        eval "$(/usr/local/bin/brew shellenv)"
        echo 'eval "$(/usr/local/bin/brew shellenv)"' >> ~/.zprofile
    fi

    if command -v brew &> /dev/null; then
        log_success "Homebrew erfolgreich installiert!"
    else
        log_error "Homebrew Installation fehlgeschlagen"
        exit 1
    fi
}

update_homebrew() {
    log_substep "Aktualisiere Homebrew..."
    brew update &> /dev/null
    log_success "Homebrew aktualisiert"
}

# ============================================================================
# OLLAMA INSTALLATION
# ============================================================================

install_ollama() {
    log_step "Installiere Ollama..."

    if command -v ollama &> /dev/null; then
        local ollama_version=$(ollama -v 2>&1 | head -n 1 || echo "unbekannt")
        log_success "Ollama bereits installiert: $ollama_version"
        return 0
    fi

    log_info "Installiere Ollama via Homebrew..."

    if brew install ollama; then
        log_success "Ollama erfolgreich installiert!"
    else
        log_error "Ollama Installation fehlgeschlagen"
        exit 1
    fi
}

start_ollama_service() {
    log_substep "Starte Ollama Service..."

    # Starte Ollama als Service
    brew services start ollama &> /dev/null || true

    # Warte bis Ollama bereit ist
    local max_attempts=30
    local attempt=0

    while [[ $attempt -lt $max_attempts ]]; do
        if curl -s http://localhost:11434/api/tags &> /dev/null; then
            log_success "Ollama Service läuft"
            return 0
        fi

        attempt=$((attempt + 1))
        sleep 1
    done

    log_warning "Ollama Service antwortet nicht, aber Installation war erfolgreich"
}

# ============================================================================
# DOCKER INSTALLATION
# ============================================================================

install_docker() {
    log_step "Installiere Docker Desktop..."

    # Prüfe ob Docker.app bereits installiert ist
    if [[ -d "/Applications/Docker.app" ]]; then
        log_success "Docker Desktop ist bereits installiert"

        # Prüfe ob Docker läuft
        if command -v docker &> /dev/null && docker info &> /dev/null 2>&1; then
            local docker_version=$(docker --version | cut -d ' ' -f 3 | tr -d ',')
            log_success "Docker läuft bereits (Version $docker_version)"
            return 0
        else
            log_info "Docker ist installiert, aber nicht gestartet"
            return 0
        fi
    fi

    # Docker Desktop ist nicht installiert - jetzt herunterladen
    log_info "Lade Docker Desktop herunter..."

    mkdir -p "$TMP_DIR"
    local docker_dmg="$TMP_DIR/Docker.dmg"

    if curl -# -L -o "$docker_dmg" "$DOCKER_DMG_URL"; then
        log_success "Download abgeschlossen"
    else
        log_error "Download fehlgeschlagen"
        exit 1
    fi

    # DMG mounten
    log_substep "Installiere Docker Desktop..."

    local mount_point=$(hdiutil attach "$docker_dmg" | grep Volumes | awk '{print $3}')

    if [[ -z "$mount_point" ]]; then
        log_error "DMG konnte nicht gemountet werden"
        exit 1
    fi

    # Docker.app kopieren
    cp -R "$mount_point/Docker.app" /Applications/

    # DMG unmounten
    hdiutil detach "$mount_point" &> /dev/null

    log_success "Docker Desktop installiert!"
}

setup_docker_cli() {
    # Prüfe ob docker Befehl bereits verfügbar ist
    if command -v docker &> /dev/null; then
        return 0
    fi

    # Docker Desktop CLI-Tools Pfad
    local docker_app_cli="/Applications/Docker.app/Contents/Resources/bin/docker"

    # Prüfe ob Docker.app CLI existiert
    if [[ ! -f "$docker_app_cli" ]]; then
        # CLI-Tools werden beim ersten Start von Docker Desktop installiert
        return 1
    fi

    # Erstelle Symlink in /usr/local/bin
    sudo mkdir -p /usr/local/bin 2>/dev/null
    sudo ln -sf "$docker_app_cli" /usr/local/bin/docker 2>/dev/null

    # Füge /usr/local/bin zum PATH hinzu
    if [[ ":$PATH:" != *":/usr/local/bin:"* ]]; then
        export PATH="/usr/local/bin:$PATH"
    fi

    # Prüfe ob docker jetzt verfügbar ist
    command -v docker &> /dev/null
}

start_docker() {
    log_substep "Starte Docker Desktop..."

    # Prüfe ob Docker bereits läuft (prüfe Socket UND CLI)
    if [[ -S /var/run/docker.sock ]] && command -v docker &> /dev/null && docker info &> /dev/null 2>&1; then
        log_success "Docker läuft bereits!"
        return 0
    fi

    # Docker starten
    if ! open -a Docker 2>/dev/null; then
        log_warning "Docker Desktop konnte nicht automatisch gestartet werden"
        log_info "Bitte starte Docker Desktop manuell:"
        log_info "  1. Öffne Programme → Docker"
        log_info "  2. Warte bis Docker läuft (Icon in der Menüleiste)"
        log_info "  3. Führe dieses Script erneut aus: ./install.sh --auto"
        echo ""
        log_info "Das Script ist idempotent - bereits installierte Komponenten werden übersprungen"
        exit 1
    fi

    # Warte kurz auf Docker Desktop Initialisierung
    log_info "Warte auf Docker Desktop Initialisierung..."
    sleep 10

    # Versuche CLI-Tools einzurichten
    if setup_docker_cli; then
        log_success "Docker CLI-Tools verfügbar"
    else
        log_info "Warte auf Installation der Docker CLI-Tools..."
    fi

    # Warte bis Docker Daemon bereit ist
    log_info "Warte auf Docker Daemon (kann 30-90 Sekunden dauern)..."

    local max_attempts=90
    local attempt=0

    while [[ $attempt -lt $max_attempts ]]; do
        # Prüfe alle 10 Sekunden ob CLI-Tools jetzt verfügbar sind
        if [[ $((attempt % 10)) -eq 0 ]]; then
            setup_docker_cli &> /dev/null
        fi

        # Prüfe ob Docker Daemon läuft
        if command -v docker &> /dev/null && docker info &> /dev/null 2>&1; then
            log_success "Docker ist bereit!"
            return 0
        fi

        print_progress $attempt $max_attempts "Docker startet..."
        attempt=$((attempt + 1))
        sleep 1
    done

    finish_progress

    # Finale Prüfung: Läuft Docker, auch wenn CLI-Befehl fehlt?
    if [[ -S /var/run/docker.sock ]] && pgrep -f "Docker.app" > /dev/null; then
        log_warning "Docker Desktop läuft, aber CLI-Tools sind noch nicht bereit"
        echo ""
        log_info "Dies kann passieren beim ersten Start von Docker Desktop."
        log_info "Lösung:"
        log_info "  1. Warte noch 30 Sekunden"
        log_info "  2. Führe das Script erneut aus: ${GREEN}./install.sh --auto${NC}"
        echo ""
        log_info "Oder öffne Docker Desktop manuell und aktiviere in Settings:"
        log_info "  → Advanced → 'Install CLI tools'"
        exit 1
    fi

    log_warning "Docker braucht länger als erwartet zum Starten"
    echo ""
    log_info "Bitte warte noch einen Moment und führe dann das Script erneut aus:"
    log_info "  ${GREEN}./install.sh --auto${NC}"
    echo ""
    log_info "Falls Docker nicht startet:"
    log_info "  1. Öffne Docker Desktop manuell (Programme → Docker)"
    log_info "  2. Warte bis das Docker-Icon in der Menüleiste erscheint"
    log_info "  3. Führe das Script erneut aus"
    echo ""
    log_info "Das Script ist idempotent - bereits installierte Komponenten werden übersprungen"
    exit 1
}

# ============================================================================
# OPEN WEBUI INSTALLATION
# ============================================================================

install_openwebui() {
    log_step "Installiere Open WebUI..."

    # Prüfe ob Container bereits existiert
    if docker ps -a --format '{{.Names}}' | grep -q "^open-webui$"; then
        log_info "Open WebUI Container existiert bereits"

        if docker ps --format '{{.Names}}' | grep -q "^open-webui$"; then
            log_success "Open WebUI läuft bereits"
            return 0
        else
            log_substep "Starte existierenden Container..."
            docker start open-webui
            log_success "Open WebUI gestartet"
            return 0
        fi
    fi

    # Pull image
    log_substep "Lade Open WebUI Image..."
    docker pull ghcr.io/open-webui/open-webui:main

    # Erstelle und starte Container
    log_substep "Erstelle Open WebUI Container..."

    docker run -d \
        -p 3000:8080 \
        --add-host=host.docker.internal:host-gateway \
        -v open-webui:/app/backend/data \
        --name open-webui \
        --restart always \
        ghcr.io/open-webui/open-webui:main

    # Warte bis Container bereit ist
    local max_attempts=30
    local attempt=0

    while [[ $attempt -lt $max_attempts ]]; do
        if curl -s http://localhost:3000 &> /dev/null; then
            log_success "Open WebUI ist bereit!"
            return 0
        fi

        attempt=$((attempt + 1))
        sleep 1
    done

    log_warning "Open WebUI antwortet nicht, aber Container läuft"
}

# ============================================================================
# MODELL DOWNLOAD
# ============================================================================

download_model() {
    if [[ "$SKIP_MODEL_DOWNLOAD" == "true" ]]; then
        log_info "Modell-Download übersprungen (--skip-model)"
        return 0
    fi

    log_step "Lade KI-Modell herunter..."

    if [[ "$AUTO_MODE" == "true" ]]; then
        local model="$DEFAULT_MODEL"
    else
        echo ""
        echo "Empfohlene Starter-Modelle:"
        echo "  1) llama3.1        (~5 GB) - Empfohlen für 16+ GB RAM"
        echo "  2) deepseek-r1:7b  (~5 GB) - Für Programmierung"
        echo "  3) gemma3:12b      (~8 GB) - Google Modell"
        echo "  4) Überspringen"
        echo ""
        read -p "Wähle eine Option (1-4) [1]: " choice

        case ${choice:-1} in
            1) model="llama3.1" ;;
            2) model="deepseek-r1:7b" ;;
            3) model="gemma3:12b" ;;
            4)
                log_info "Modell-Download übersprungen"
                return 0
                ;;
            *)
                log_warning "Ungültige Auswahl, verwende llama3.1"
                model="llama3.1"
                ;;
        esac
    fi

    log_info "Lade Modell: $model"
    log_info "Dies kann 5-10 Minuten dauern..."

    if ollama pull "$model"; then
        log_success "Modell $model erfolgreich heruntergeladen!"
    else
        log_warning "Modell-Download fehlgeschlagen"
        log_info "Du kannst das Modell später mit 'ollama pull $model' herunterladen"
    fi
}

# ============================================================================
# CLEANUP
# ============================================================================

cleanup() {
    if [[ -d "$TMP_DIR" ]]; then
        rm -rf "$TMP_DIR"
    fi
}

trap cleanup EXIT

# ============================================================================
# ABSCHLUSS
# ============================================================================

show_completion() {
    echo ""
    echo -e "${GREEN}${BOLD}"
    echo "╔════════════════════════════════════════════════════════════════╗"
    echo "║                                                                ║"
    echo "║          ${SPARKLE} Installation erfolgreich abgeschlossen! ${SPARKLE}            ║"
    echo "║                                                                ║"
    echo "╚════════════════════════════════════════════════════════════════╝"
    echo -e "${NC}\n"

    log_success "Ollama läuft und ist bereit"
    log_success "Docker Desktop läuft"
    log_success "Open WebUI ist verfügbar"

    echo ""
    echo -e "${CYAN}${BOLD}${ARROW} Nächste Schritte:${NC}\n"

    echo -e "  ${BOLD}1.${NC} Öffne Open WebUI im Browser:"
    echo -e "     ${BLUE}http://localhost:3000${NC}"
    echo ""

    echo -e "  ${BOLD}2.${NC} Erstelle einen Administrator-Account"
    echo ""

    echo -e "  ${BOLD}3.${NC} Lade weitere Modelle (optional):"
    echo -e "     ${DIM}ollama pull deepseek-r1:32b${NC}"
    echo -e "     ${DIM}ollama pull gemma3:12b${NC}"
    echo ""

    echo -e "  ${BOLD}4.${NC} Verwaltungs-Tools:"
    echo -e "     ${DIM}./scripts/manage.sh  ${NC} - System verwalten"
    echo -e "     ${DIM}./scripts/update.sh  ${NC} - Updates installieren"
    echo ""

    echo -e "${MAGENTA}${BOLD}Dokumentation:${NC} docs/GETTING_STARTED.md"
    echo -e "${MAGENTA}${BOLD}Modell-Guide:${NC}  docs/MODEL_RECOMMENDATIONS.md"
    echo ""

    # Browser öffnen - AKTIVE Wahl erforderlich
    echo ""
    echo -e "${CYAN}${BOLD}Open WebUI ist bereit unter: ${BLUE}http://localhost:3000${NC}"
    echo ""
    echo "Was möchtest du tun?"
    echo ""
    echo "  1) Open WebUI jetzt im Browser öffnen"
    echo "  2) Zurück (URL manuell später öffnen)"
    echo ""

    read -p "Wähle eine Option (1/2): " -n 1 -r
    echo ""
    echo ""

    case $REPLY in
        1)
            open http://localhost:3000
            log_success "Browser geöffnet: http://localhost:3000"
            echo ""
            sleep 2
            ;;
        2)
            log_info "Browser nicht geöffnet"
            log_info "Starte später manuell: http://localhost:3000"
            echo ""
            ;;
        *)
            log_warning "Ungültige Auswahl"
            log_info "Starte manuell: http://localhost:3000"
            echo ""
            ;;
    esac

    echo ""
    echo -e "${GREEN}${BOLD}Viel Spaß mit deinem lokalen KI-System! ${ROCKET}${NC}\n"
}

# ============================================================================
# DISCLAIMER
# ============================================================================

show_disclaimer() {
    clear

    if [[ -f "LICENSE_DISCLAIMER.txt" ]]; then
        cat LICENSE_DISCLAIMER.txt
    else
        echo -e "${RED}${BOLD}WICHTIGER HAFTUNGSAUSSCHLUSS${NC}\n"
        echo "Diese Software wird OHNE JEGLICHE GARANTIE bereitgestellt."
        echo "Die Nutzung erfolgt AUSSCHLIEẞLICH AUF EIGENE GEFAHR."
        echo ""
        echo "Der Autor übernimmt KEINERLEI Haftung für:"
        echo "  • Datenverlust"
        echo "  • Systemabstürze"
        echo "  • Konfigurationsänderungen"
        echo "  • Jegliche Schäden"
        echo ""
        echo "ERSTELLEN SIE EIN BACKUP VOR DER INSTALLATION!"
    fi

    echo ""
    echo -e "${YELLOW}${BOLD}═══════════════════════════════════════════════════════════════════${NC}"
    echo ""

    if [[ "$AUTO_MODE" == "true" ]]; then
        log_warning "Automatischer Modus: Disclaimer wird automatisch akzeptiert"
        echo ""
        log_info "Warte 5 Sekunden..."
        sleep 5
        return 0
    fi

    echo -e "${RED}${BOLD}Durch Fortfahren bestätigen Sie:${NC}"
    echo ""
    echo "  ✓ Sie haben den Haftungsausschluss gelesen und verstanden"
    echo "  ✓ Sie haben ein aktuelles Backup Ihres Systems erstellt"
    echo "  ✓ Sie akzeptieren die Nutzung auf eigene Gefahr"
    echo "  ✓ Sie übernehmen die volle Verantwortung"
    echo "  ✓ Sie verzichten auf jegliche Haftungsansprüche"
    echo ""

    read -p "Akzeptieren Sie diese Bedingungen? (JA/nein): " -r REPLY
    echo ""

    # Explizite Bestätigung mit "JA" (case-insensitive)
    if [[ "${REPLY^^}" == "JA" ]] || [[ "${REPLY^^}" == "YES" ]]; then
        log_success "Bedingungen akzeptiert"
        echo ""
        sleep 1
        return 0
    else
        log_error "Bedingungen nicht akzeptiert"
        log_info "Installation abgebrochen"
        echo ""
        exit 0
    fi
}

# ============================================================================
# MAIN
# ============================================================================

main() {
    parse_arguments "$@"

    print_header

    # Zeige Disclaimer (außer bei --help)
    show_disclaimer

    if [[ "$AUTO_MODE" == "true" ]]; then
        log_info "Automatischer Modus aktiviert"
        echo ""
    fi

    # System-Checks
    check_macos
    check_architecture
    check_memory
    check_disk_space
    check_internet

    echo ""

    # Bestätigung
    if [[ "$AUTO_MODE" != "true" ]]; then
        log_info "Die Installation wird folgende Komponenten installieren:"
        echo "  • Homebrew"
        echo "  • Ollama"
        echo "  • Docker Desktop"
        echo "  • Open WebUI"
        echo ""

        if ! ask_confirmation "Installation starten?" "y"; then
            log_info "Installation abgebrochen"
            exit 0
        fi

        echo ""
    fi

    # Installation
    install_homebrew
    update_homebrew
    install_ollama
    start_ollama_service
    install_docker
    start_docker
    install_openwebui
    download_model

    # Abschluss
    show_completion
}

# Start
main "$@"
