#!/bin/bash

###############################################################################
# OllamaDeploy Management Script
# Verwaltung von Ollama, Docker und Open WebUI
###############################################################################

# Farben
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_header() {
    clear
    echo -e "${BLUE}"
    echo "╔════════════════════════════════════════════════════════╗"
    echo "║         OllamaDeploy Verwaltung v1.0                   ║"
    echo "╚════════════════════════════════════════════════════════╝"
    echo -e "${NC}"
}

print_step() {
    echo -e "${GREEN}[✓]${NC} $1"
}

print_info() {
    echo -e "${BLUE}[ℹ]${NC} $1"
}

print_error() {
    echo -e "${RED}[✗]${NC} $1"
}

# Status-Anzeige
show_status() {
    print_header
    echo -e "${YELLOW}═══ System Status ═══${NC}\n"

    # Ollama Status
    if command -v ollama &> /dev/null; then
        OLLAMA_VERSION=$(ollama -v 2>&1 | head -n 1)
        print_step "Ollama: ${OLLAMA_VERSION}"
    else
        print_error "Ollama: Nicht installiert"
    fi

    # Docker Status
    if command -v docker &> /dev/null; then
        if docker info &> /dev/null 2>&1; then
            DOCKER_VERSION=$(docker --version | cut -d ' ' -f 3 | tr -d ',')
            print_step "Docker: Version $DOCKER_VERSION (läuft)"
        else
            print_error "Docker: Installiert aber läuft nicht"
        fi
    else
        print_error "Docker: Nicht installiert"
    fi

    # Open WebUI Status
    if docker ps --format '{{.Names}}' | grep -q "^open-webui$"; then
        print_step "Open WebUI: Läuft auf http://localhost:3000"
    elif docker ps -a --format '{{.Names}}' | grep -q "^open-webui$"; then
        print_error "Open WebUI: Gestoppt"
    else
        print_error "Open WebUI: Nicht installiert"
    fi

    echo ""
    echo -e "${YELLOW}═══ Installierte Modelle ═══${NC}\n"

    if command -v ollama &> /dev/null; then
        ollama list
    else
        print_error "Ollama nicht verfügbar"
    fi

    echo ""
}

# Modell-Verwaltung
manage_models() {
    print_header
    echo -e "${YELLOW}═══ Modell-Verwaltung ═══${NC}\n"
    echo "1) Neues Modell herunterladen"
    echo "2) Modell löschen"
    echo "3) Modell im Terminal testen"
    echo "4) Zurück zum Hauptmenü"
    echo ""
    read -p "Wähle eine Option (1-4): " -n 1 -r
    echo ""

    case $REPLY in
        1)
            download_model
            ;;
        2)
            remove_model
            ;;
        3)
            test_model
            ;;
        4)
            return
            ;;
        *)
            print_error "Ungültige Auswahl"
            sleep 2
            ;;
    esac
}

download_model() {
    echo ""
    echo -e "${YELLOW}Beliebte Modelle:${NC}"
    echo ""
    echo "Kleine Modelle (8-16 GB RAM):"
    echo "  • llama3.1         - 5 GB"
    echo "  • deepseek-r1:7b   - 5 GB"
    echo "  • gemma3:12b       - 8 GB"
    echo ""
    echo "Mittlere Modelle (32+ GB RAM):"
    echo "  • llama3.3:70b     - 43 GB"
    echo "  • deepseek-r1:32b  - 20 GB"
    echo ""
    echo "Große Modelle (128+ GB RAM):"
    echo "  • deepseek-r1:671b - 404 GB"
    echo ""
    echo "Suche weitere Modelle auf: https://ollama.com/search"
    echo ""
    read -p "Modell-Name eingeben (oder Enter für Abbruch): " MODEL_NAME

    if [ -z "$MODEL_NAME" ]; then
        print_info "Abgebrochen"
        sleep 1
        return
    fi

    print_info "Lade $MODEL_NAME herunter..."
    if ollama pull "$MODEL_NAME"; then
        print_step "Download erfolgreich!"
    else
        print_error "Download fehlgeschlagen"
    fi

    echo ""
    read -p "Drücke Enter zum Fortfahren..."
}

remove_model() {
    echo ""
    print_info "Installierte Modelle:"
    ollama list
    echo ""
    read -p "Modell-Name zum Löschen (oder Enter für Abbruch): " MODEL_NAME

    if [ -z "$MODEL_NAME" ]; then
        print_info "Abgebrochen"
        sleep 1
        return
    fi

    read -p "Wirklich $MODEL_NAME löschen? (y/n): " -n 1 -r
    echo ""

    if [[ $REPLY =~ ^[Yy]$ ]]; then
        if ollama rm "$MODEL_NAME"; then
            print_step "Modell gelöscht!"
        else
            print_error "Fehler beim Löschen"
        fi
    else
        print_info "Abgebrochen"
    fi

    echo ""
    read -p "Drücke Enter zum Fortfahren..."
}

test_model() {
    echo ""
    print_info "Installierte Modelle:"
    ollama list
    echo ""
    read -p "Modell-Name zum Testen: " MODEL_NAME

    if [ -z "$MODEL_NAME" ]; then
        print_info "Abgebrochen"
        sleep 1
        return
    fi

    print_info "Starte $MODEL_NAME (Mit /bye beenden)..."
    echo ""
    ollama run "$MODEL_NAME"

    echo ""
    read -p "Drücke Enter zum Fortfahren..."
}

# Container-Verwaltung
manage_containers() {
    print_header
    echo -e "${YELLOW}═══ Container-Verwaltung ═══${NC}\n"
    echo "1) Open WebUI starten"
    echo "2) Open WebUI stoppen"
    echo "3) Open WebUI neu starten"
    echo "4) Open WebUI Logs anzeigen"
    echo "5) Open WebUI entfernen"
    echo "6) Zurück zum Hauptmenü"
    echo ""
    read -p "Wähle eine Option (1-6): " -n 1 -r
    echo ""

    case $REPLY in
        1)
            docker start open-webui && print_step "Open WebUI gestartet"
            ;;
        2)
            docker stop open-webui && print_step "Open WebUI gestoppt"
            ;;
        3)
            docker restart open-webui && print_step "Open WebUI neu gestartet"
            ;;
        4)
            docker logs -f open-webui
            ;;
        5)
            read -p "Wirklich löschen? Daten bleiben erhalten (y/n): " -n 1 -r
            echo ""
            if [[ $REPLY =~ ^[Yy]$ ]]; then
                docker rm -f open-webui && print_step "Container gelöscht"
            fi
            ;;
        6)
            return
            ;;
        *)
            print_error "Ungültige Auswahl"
            ;;
    esac

    echo ""
    read -p "Drücke Enter zum Fortfahren..."
}

# System-Informationen
show_system_info() {
    print_header
    echo -e "${YELLOW}═══ System-Informationen ═══${NC}\n"

    # Hardware Info
    echo -e "${BLUE}Hardware:${NC}"
    echo "  CPU: $(sysctl -n machdep.cpu.brand_string)"
    echo "  Arch: $(uname -m)"
    TOTAL_MEM=$(sysctl -n hw.memsize)
    TOTAL_GB=$((TOTAL_MEM / 1024 / 1024 / 1024))
    echo "  RAM: ${TOTAL_GB} GB"
    echo ""

    # Speicher Info
    echo -e "${BLUE}Speicher:${NC}"
    df -h / | tail -1 | awk '{print "  Gesamt: "$2"\n  Benutzt: "$3"\n  Frei: "$4"\n  Auslastung: "$5}'
    echo ""

    # Ollama Speicher
    if [ -d ~/.ollama ]; then
        OLLAMA_SIZE=$(du -sh ~/.ollama 2>/dev/null | cut -f1)
        echo -e "${BLUE}Ollama Modelle:${NC}"
        echo "  Speicherort: ~/.ollama"
        echo "  Größe: ${OLLAMA_SIZE}"
        echo ""
    fi

    # Docker Speicher
    if command -v docker &> /dev/null && docker info &> /dev/null 2>&1; then
        echo -e "${BLUE}Docker:${NC}"
        docker system df
        echo ""
    fi

    # Netzwerk Info
    echo -e "${BLUE}Netzwerk-Zugriff:${NC}"
    LOCAL_IP=$(ipconfig getifaddr en0 2>/dev/null || ipconfig getifaddr en1 2>/dev/null || echo "Nicht verfügbar")
    echo "  Lokale IP: ${LOCAL_IP}"
    echo "  Open WebUI: http://${LOCAL_IP}:3000"
    echo "  (von anderen Geräten im Netzwerk erreichbar)"
    echo ""

    read -p "Drücke Enter zum Fortfahren..."
}

# Aufräumen
cleanup() {
    print_header
    echo -e "${YELLOW}═══ System aufräumen ═══${NC}\n"
    echo "1) Docker Cache leeren"
    echo "2) Ungenutzte Modelle anzeigen"
    echo "3) Vollständige Bereinigung"
    echo "4) Zurück zum Hauptmenü"
    echo ""
    read -p "Wähle eine Option (1-4): " -n 1 -r
    echo ""

    case $REPLY in
        1)
            print_info "Räume Docker auf..."
            docker system prune -f
            print_step "Fertig!"
            ;;
        2)
            print_info "Installierte Modelle:"
            ollama list
            ;;
        3)
            read -p "Wirklich alles bereinigen? (y/n): " -n 1 -r
            echo ""
            if [[ $REPLY =~ ^[Yy]$ ]]; then
                print_info "Räume auf..."
                docker system prune -a -f --volumes
                print_step "Fertig!"
            fi
            ;;
        4)
            return
            ;;
        *)
            print_error "Ungültige Auswahl"
            ;;
    esac

    echo ""
    read -p "Drücke Enter zum Fortfahren..."
}

# Hauptmenü
main_menu() {
    while true; do
        show_status
        echo -e "${YELLOW}═══ Hauptmenü ═══${NC}\n"
        echo "1) Modell-Verwaltung"
        echo "2) Container-Verwaltung"
        echo "3) System-Informationen"
        echo "4) System aufräumen"
        echo "5) Open WebUI öffnen"
        echo "6) Beenden"
        echo ""
        read -p "Wähle eine Option (1-6): " -n 1 -r
        echo ""

        case $REPLY in
            1)
                manage_models
                ;;
            2)
                manage_containers
                ;;
            3)
                show_system_info
                ;;
            4)
                cleanup
                ;;
            5)
                print_info "Öffne http://localhost:3000 im Browser..."
                open http://localhost:3000
                sleep 1
                ;;
            6)
                print_info "Auf Wiedersehen!"
                exit 0
                ;;
            *)
                print_error "Ungültige Auswahl"
                sleep 2
                ;;
        esac
    done
}

# Starte Programm
main_menu
