#!/bin/bash

###############################################################################
# OllamaDeploy Setup Script
# Automatische Installation von Ollama, Docker Desktop und Open WebUI
###############################################################################

set -e  # Beende bei Fehler

# Farben für Output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Funktionen
print_header() {
    echo -e "${BLUE}"
    echo "╔════════════════════════════════════════════════════════╗"
    echo "║           OllamaDeploy Setup v1.0                      ║"
    echo "║    Lokales KI-System für Mac mit Apple Silicon         ║"
    echo "╚════════════════════════════════════════════════════════╝"
    echo -e "${NC}"
}

print_step() {
    echo -e "${GREEN}[✓]${NC} $1"
}

print_info() {
    echo -e "${BLUE}[ℹ]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[⚠]${NC} $1"
}

print_error() {
    echo -e "${RED}[✗]${NC} $1"
}

check_macos() {
    if [[ "$OSTYPE" != "darwin"* ]]; then
        print_error "Dieses Skript läuft nur auf macOS!"
        exit 1
    fi
    print_step "macOS erkannt"
}

check_apple_silicon() {
    ARCH=$(uname -m)
    if [[ "$ARCH" != "arm64" ]]; then
        print_warning "Warnung: Kein Apple Silicon erkannt (arm64)"
        print_info "Erkannte Architektur: $ARCH"
        read -p "Möchtest du trotzdem fortfahren? (y/n) " -n 1 -r
        echo
        if [[ ! $REPLY =~ ^[Yy]$ ]]; then
            exit 1
        fi
    else
        print_step "Apple Silicon (M1/M2/M3/M4) erkannt"
    fi
}

check_memory() {
    TOTAL_MEM=$(sysctl -n hw.memsize)
    TOTAL_GB=$((TOTAL_MEM / 1024 / 1024 / 1024))

    print_info "Installierter RAM: ${TOTAL_GB} GB"

    if [ "$TOTAL_GB" -lt 16 ]; then
        print_warning "Warnung: Weniger als 16 GB RAM erkannt"
        print_info "Für beste Ergebnisse werden mindestens 16 GB empfohlen"
    else
        print_step "Ausreichend RAM vorhanden"
    fi
}

check_disk_space() {
    FREE_SPACE=$(df -g / | tail -1 | awk '{print $4}')
    print_info "Freier Speicherplatz: ${FREE_SPACE} GB"

    if [ "$FREE_SPACE" -lt 50 ]; then
        print_warning "Warnung: Weniger als 50 GB freier Speicher"
        print_info "Für KI-Modelle werden mindestens 50 GB empfohlen"
    else
        print_step "Ausreichend Speicherplatz vorhanden"
    fi
}

check_ollama() {
    if command -v ollama &> /dev/null; then
        OLLAMA_VERSION=$(ollama -v 2>&1 | head -n 1 || echo "unbekannt")
        print_step "Ollama ist bereits installiert: $OLLAMA_VERSION"
        return 0
    fi
    return 1
}

install_ollama() {
    print_info "Lade Ollama herunter..."

    if [ ! -d "/Applications/Ollama.app" ]; then
        print_info "Bitte lade Ollama manuell herunter von:"
        print_info "  https://ollama.com/download"
        print_info ""
        print_info "Oder verwende Homebrew:"
        print_info "  brew install ollama"
        print_info ""
        read -p "Drücke Enter, wenn Ollama installiert ist..."

        if ! command -v ollama &> /dev/null; then
            print_error "Ollama wurde nicht gefunden!"
            exit 1
        fi
    fi

    print_step "Ollama erfolgreich installiert"
}

check_docker() {
    if command -v docker &> /dev/null; then
        DOCKER_VERSION=$(docker --version | cut -d ' ' -f 3 | tr -d ',')
        print_step "Docker ist bereits installiert: $DOCKER_VERSION"

        # Prüfe ob Docker läuft
        if ! docker info &> /dev/null; then
            print_warning "Docker läuft nicht. Starte Docker Desktop..."
            open -a Docker
            print_info "Warte auf Docker..."
            sleep 10
        fi
        return 0
    fi
    return 1
}

install_docker() {
    print_info "Docker Desktop muss manuell installiert werden"
    print_info "Bitte lade Docker Desktop herunter von:"
    print_info "  https://www.docker.com/products/docker-desktop"
    print_info ""
    read -p "Drücke Enter, wenn Docker Desktop installiert ist..."

    if ! command -v docker &> /dev/null; then
        print_error "Docker wurde nicht gefunden!"
        exit 1
    fi

    # Starte Docker falls noch nicht gestartet
    if ! docker info &> /dev/null; then
        print_info "Starte Docker Desktop..."
        open -a Docker
        print_info "Warte auf Docker (kann 30-60 Sekunden dauern)..."

        # Warte bis Docker bereit ist
        for i in {1..30}; do
            if docker info &> /dev/null 2>&1; then
                print_step "Docker ist bereit"
                break
            fi
            echo -n "."
            sleep 2
        done
        echo ""
    fi

    print_step "Docker Desktop erfolgreich installiert"
}

install_openwebui() {
    print_info "Installiere Open WebUI Container..."

    # Prüfe ob Container bereits existiert
    if docker ps -a --format '{{.Names}}' | grep -q "^open-webui$"; then
        print_info "Open WebUI Container existiert bereits"

        # Prüfe ob Container läuft
        if docker ps --format '{{.Names}}' | grep -q "^open-webui$"; then
            print_step "Open WebUI läuft bereits"
            return 0
        else
            print_info "Starte existierenden Container..."
            docker start open-webui
            print_step "Open WebUI gestartet"
            return 0
        fi
    fi

    # Erstelle und starte neuen Container
    print_info "Erstelle Open WebUI Container..."
    docker run -d \
        -p 3000:8080 \
        --add-host=host.docker.internal:host-gateway \
        -v open-webui:/app/backend/data \
        --name open-webui \
        --restart always \
        ghcr.io/open-webui/open-webui:main

    print_step "Open WebUI erfolgreich installiert"
}

download_starter_model() {
    print_info "Möchtest du ein KI-Modell herunterladen?"
    echo ""
    echo "Empfohlene Starter-Modelle:"
    echo "  1) llama3.1        (8B Parameter,  ~5 GB) - Empfohlen für 16+ GB RAM"
    echo "  2) deepseek-r1:7b  (7B Parameter,  ~5 GB) - Schnell und effizient"
    echo "  3) gemma3:12b      (12B Parameter, ~8 GB) - Google Modell"
    echo "  4) Überspringen"
    echo ""
    read -p "Wähle eine Option (1-4): " -n 1 -r MODEL_CHOICE
    echo ""

    case $MODEL_CHOICE in
        1)
            print_info "Lade llama3.1 herunter..."
            ollama pull llama3.1
            print_step "llama3.1 erfolgreich heruntergeladen"
            ;;
        2)
            print_info "Lade deepseek-r1:7b herunter..."
            ollama pull deepseek-r1:7b
            print_step "deepseek-r1:7b erfolgreich heruntergeladen"
            ;;
        3)
            print_info "Lade gemma3:12b herunter..."
            ollama pull gemma3:12b
            print_step "gemma3:12b erfolgreich heruntergeladen"
            ;;
        4)
            print_info "Überspringe Model-Download"
            ;;
        *)
            print_warning "Ungültige Auswahl, überspringe Download"
            ;;
    esac
}

show_completion() {
    echo ""
    echo -e "${GREEN}"
    echo "╔════════════════════════════════════════════════════════╗"
    echo "║               Installation abgeschlossen!               ║"
    echo "╚════════════════════════════════════════════════════════╝"
    echo -e "${NC}"
    echo ""
    print_step "Ollama läuft und ist bereit"
    print_step "Docker Desktop läuft"
    print_step "Open WebUI ist verfügbar"
    echo ""
    print_info "Nächste Schritte:"
    echo ""
    echo "  1. Öffne Open WebUI im Browser:"
    echo "     ${BLUE}http://localhost:3000${NC}"
    echo ""
    echo "  2. Erstelle einen Administrator-Account"
    echo ""
    echo "  3. Lade weitere Modelle herunter (optional):"
    echo "     ${YELLOW}ollama pull llama3.1${NC}"
    echo "     ${YELLOW}ollama pull deepseek-r1:32b${NC}"
    echo ""
    echo "  4. Nutze die Verwaltungs-Skripte:"
    echo "     ${YELLOW}./scripts/manage.sh${NC} - Verwalte Container und Modelle"
    echo "     ${YELLOW}./scripts/update.sh${NC}  - Update Komponenten"
    echo ""
    print_info "Dokumentation findest du in: ${BLUE}docs/${NC}"
    echo ""
}

# Hauptprogramm
main() {
    print_header

    print_info "Prüfe Systemanforderungen..."
    check_macos
    check_apple_silicon
    check_memory
    check_disk_space

    echo ""
    print_info "Installation wird gestartet..."
    echo ""

    # Ollama installieren
    if ! check_ollama; then
        install_ollama
    fi

    # Docker installieren
    if ! check_docker; then
        install_docker
    fi

    # Open WebUI installieren
    install_openwebui

    echo ""

    # Starter-Modell herunterladen
    download_starter_model

    # Abschluss
    show_completion
}

# Starte Installation
main
