#!/bin/bash

###############################################################################
# OllamaDeploy Update Script
# Aktualisierung von Ollama, Docker und Open WebUI
###############################################################################

# Farben
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_header() {
    echo -e "${BLUE}"
    echo "╔════════════════════════════════════════════════════════╗"
    echo "║         OllamaDeploy Update v1.0                       ║"
    echo "╚════════════════════════════════════════════════════════╝"
    echo -e "${NC}"
}

print_step() {
    echo -e "${GREEN}[✓]${NC} $1"
}

print_info() {
    echo -e "${BLUE}[ℹ]${NC} $1"
}

print_error() {
    echo -e "${RED}[✗]${NC} $1"
}

update_ollama() {
    print_info "Prüfe Ollama Updates..."

    if ! command -v ollama &> /dev/null; then
        print_error "Ollama ist nicht installiert"
        return 1
    fi

    CURRENT_VERSION=$(ollama -v 2>&1 | grep -oE '[0-9]+\.[0-9]+\.[0-9]+' | head -n 1)
    print_info "Installierte Ollama Version: $CURRENT_VERSION"

    print_info "Bitte besuche https://ollama.com/download für die neueste Version"
    print_info "Oder verwende: brew upgrade ollama"

    read -p "Möchtest du Ollama jetzt über Homebrew aktualisieren? (y/n): " -n 1 -r
    echo ""

    if [[ $REPLY =~ ^[Yy]$ ]]; then
        if command -v brew &> /dev/null; then
            brew upgrade ollama
            print_step "Ollama aktualisiert"
        else
            print_error "Homebrew ist nicht installiert"
            print_info "Installiere Homebrew: /bin/bash -c \"\$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)\""
        fi
    fi
}

update_docker() {
    print_info "Prüfe Docker Updates..."

    if ! command -v docker &> /dev/null; then
        print_error "Docker ist nicht installiert"
        return 1
    fi

    CURRENT_VERSION=$(docker --version | cut -d ' ' -f 3 | tr -d ',')
    print_info "Installierte Docker Version: $CURRENT_VERSION"

    print_info "Docker Desktop prüft automatisch auf Updates"
    print_info "Öffne Docker Desktop → Einstellungen → Software Updates"

    read -p "Docker Desktop jetzt öffnen? (y/n): " -n 1 -r
    echo ""

    if [[ $REPLY =~ ^[Yy]$ ]]; then
        open -a Docker
    fi
}

update_openwebui() {
    print_info "Aktualisiere Open WebUI..."

    if ! docker ps -a --format '{{.Names}}' | grep -q "^open-webui$"; then
        print_error "Open WebUI Container nicht gefunden"
        return 1
    fi

    # Stoppe Container
    print_info "Stoppe Open WebUI..."
    docker stop open-webui

    # Entferne alten Container
    print_info "Entferne alten Container..."
    docker rm open-webui

    # Pull neues Image
    print_info "Lade neueste Version..."
    docker pull ghcr.io/open-webui/open-webui:main

    # Starte neuen Container
    print_info "Starte aktualisierten Container..."
    docker run -d \
        -p 3000:8080 \
        --add-host=host.docker.internal:host-gateway \
        -v open-webui:/app/backend/data \
        --name open-webui \
        --restart always \
        ghcr.io/open-webui/open-webui:main

    print_step "Open WebUI erfolgreich aktualisiert"
}

update_models() {
    print_info "Verfügbare Modelle aktualisieren..."

    if ! command -v ollama &> /dev/null; then
        print_error "Ollama ist nicht installiert"
        return 1
    fi

    print_info "Installierte Modelle:"
    ollama list

    echo ""
    read -p "Möchtest du alle Modelle aktualisieren? (y/n): " -n 1 -r
    echo ""

    if [[ $REPLY =~ ^[Yy]$ ]]; then
        # Liste alle Modellnamen
        MODELS=$(ollama list | tail -n +2 | awk '{print $1}')

        for MODEL in $MODELS; do
            print_info "Aktualisiere $MODEL..."
            ollama pull "$MODEL"
        done

        print_step "Alle Modelle aktualisiert"
    fi
}

show_update_menu() {
    print_header
    echo ""
    echo "Was möchtest du aktualisieren?"
    echo ""
    echo "1) Ollama"
    echo "2) Docker Desktop"
    echo "3) Open WebUI"
    echo "4) KI-Modelle"
    echo "5) Alles"
    echo "6) Beenden"
    echo ""
    read -p "Wähle eine Option (1-6): " -n 1 -r
    echo ""

    case $REPLY in
        1)
            update_ollama
            ;;
        2)
            update_docker
            ;;
        3)
            update_openwebui
            ;;
        4)
            update_models
            ;;
        5)
            update_ollama
            echo ""
            update_docker
            echo ""
            update_openwebui
            echo ""
            update_models
            ;;
        6)
            print_info "Auf Wiedersehen!"
            exit 0
            ;;
        *)
            print_error "Ungültige Auswahl"
            exit 1
            ;;
    esac
}

# Hauptprogramm
show_update_menu

echo ""
print_step "Update-Prozess abgeschlossen"
echo ""
