#!/bin/bash

###############################################################################
# OllamaDeploy - Interaktives Hauptmenü
# Starte, verwalte oder deinstalliere das lokale KI-System
###############################################################################

# Kein set -e hier, da interaktives Menü bei Fehlern nicht abbrechen soll

# ============================================================================
# FARBEN UND SYMBOLE
# ============================================================================

if [[ -t 1 ]]; then
    RED='\033[0;31m'
    GREEN='\033[0;32m'
    YELLOW='\033[1;33m'
    BLUE='\033[0;34m'
    MAGENTA='\033[0;35m'
    CYAN='\033[0;36m'
    BOLD='\033[1m'
    DIM='\033[2m'
    NC='\033[0m'
else
    RED=''
    GREEN=''
    YELLOW=''
    BLUE=''
    MAGENTA=''
    CYAN=''
    BOLD=''
    DIM=''
    NC=''
fi

ARROW="→"
ROCKET="🚀"

# ============================================================================
# MENÜ-FUNKTIONEN
# ============================================================================

print_header() {
    clear
    echo -e "${BLUE}${BOLD}"
    echo "╔════════════════════════════════════════════════════════════════╗"
    echo "║                                                                ║"
    echo "║        ${ROCKET}  OllamaDeploy - Lokales KI-System  ${ROCKET}               ║"
    echo "║                                                                ║"
    echo "╚════════════════════════════════════════════════════════════════╝"
    echo -e "${NC}\n"
}

# Menü mit Pfeiltasten-Navigation
show_menu() {
    local selected=$1
    local options=(
        "Vollautomatische Installation (empfohlen)"
        "Interaktive Installation mit Rückfragen"
        "Installation ohne Modell-Download"
        "System verwalten (Container, Modelle, Status)"
        "System aktualisieren (Updates)"
        "Open WebUI im Browser öffnen"
        "Schnellstart-Guide anzeigen"
        "Deinstallation"
        "Beenden"
    )

    local descriptions=(
        "Installiert alles automatisch ohne Rückfragen (~10 Min.)"
        "Installation mit Bestätigungen und Auswahlmöglichkeiten"
        "Schnellere Installation, Modelle später herunterladen"
        "Interaktives Tool zur Verwaltung aller Komponenten"
        "Aktualisiere Ollama, Docker oder Open WebUI"
        "Öffnet http://localhost:3000 im Browser"
        "Zeigt häufige Befehle und Quick Actions"
        "Entfernt alle installierten Komponenten"
        "Menü verlassen"
    )

    echo -e "${CYAN}${BOLD}Was möchtest du tun?${NC}\n"

    for i in "${!options[@]}"; do
        if [ $i -eq $selected ]; then
            echo -e "${GREEN}${BOLD}${ARROW} ${options[$i]}${NC}"
            echo -e "  ${DIM}${descriptions[$i]}${NC}\n"
        else
            echo -e "  ${options[$i]}"
            echo -e "  ${DIM}${descriptions[$i]}${NC}\n"
        fi
    done

    echo -e "${DIM}Nutze ↑/↓ Pfeiltasten zum Navigieren, Enter zum Auswählen${NC}"
}

# Tastatureingabe ohne Enter
get_key() {
    local key
    IFS= read -rsn1 key 2>/dev/null >&2

    # Escape-Sequenzen für Pfeiltasten
    if [[ $key == $'\x1b' ]]; then
        read -rsn2 key 2>/dev/null >&2
        case $key in
            '[A') echo "up" ;;
            '[B') echo "down" ;;
        esac
    elif [[ $key == "" ]]; then
        echo "enter"
    fi
}

# ============================================================================
# AKTIONEN
# ============================================================================

action_auto_install() {
    clear
    echo -e "${GREEN}${BOLD}Starte vollautomatische Installation...${NC}\n"
    sleep 1
    ./install.sh --auto || true
    echo ""
    read -p "Drücke Enter um zum Menü zurückzukehren..."
}

action_interactive_install() {
    clear
    echo -e "${GREEN}${BOLD}Starte interaktive Installation...${NC}\n"
    sleep 1
    ./install.sh || true
    echo ""
    read -p "Drücke Enter um zum Menü zurückzukehren..."
}

action_skip_model_install() {
    clear
    echo -e "${GREEN}${BOLD}Starte Installation ohne Modell-Download...${NC}\n"
    sleep 1
    ./install.sh --auto --skip-model || true
    echo ""
    read -p "Drücke Enter um zum Menü zurückzukehren..."
}

action_manage() {
    clear
    ./scripts/manage.sh || true
    # manage.sh hat eigene Menü-Loop, kehrt erst zurück wenn User "Beenden" wählt
}

action_update() {
    clear
    ./scripts/update.sh || true
    echo ""
    read -p "Drücke Enter um zum Menü zurückzukehren..."
}

action_open_webui() {
    clear
    echo -e "${BLUE}[ℹ]${NC} Öffne Open WebUI im Browser..."

    # Prüfe ob Container läuft
    if command -v docker &> /dev/null && docker ps --format '{{.Names}}' | grep -q "^open-webui$"; then
        open http://localhost:3000
        echo -e "${GREEN}[✓]${NC} Browser geöffnet: http://localhost:3000"
    else
        echo -e "${YELLOW}[⚠]${NC} Open WebUI läuft nicht!"
        echo -e "${BLUE}[ℹ]${NC} Starte zuerst die Installation oder verwalte das System"
    fi

    echo ""
    read -p "Drücke Enter zum Fortfahren..."
}

action_quick_start() {
    clear
    ./scripts/quick-start.sh || true
    echo ""
    read -p "Drücke Enter um zum Menü zurückzukehren..."
}

action_uninstall() {
    clear
    echo -e "${YELLOW}${BOLD}Deinstallation${NC}\n"
    echo "Wie möchtest du deinstallieren?"
    echo ""
    echo "1) Interaktiv (mit Rückfragen)"
    echo "2) Vollständig ohne Rückfragen (--force)"
    echo "3) Software entfernen, Daten behalten (--keep-data)"
    echo "4) Zurück zum Hauptmenü"
    echo ""
    read -p "Wähle eine Option (1-4): " -n 1 -r
    echo ""

    case $REPLY in
        1)
            ./uninstall.sh || true
            ;;
        2)
            ./uninstall.sh --force || true
            ;;
        3)
            ./uninstall.sh --keep-data || true
            ;;
        4)
            return
            ;;
        *)
            echo -e "${RED}[✗]${NC} Ungültige Auswahl"
            sleep 2
            return
            ;;
    esac

    echo ""
    read -p "Drücke Enter um zum Menü zurückzukehren..."
}

# ============================================================================
# HAUPTMENÜ
# ============================================================================

main_menu() {
    local selected=0
    local num_options=9

    while true; do
        print_header
        show_menu $selected

        key=$(get_key)

        case $key in
            up)
                ((selected--))
                if [ $selected -lt 0 ]; then
                    selected=$((num_options - 1))
                fi
                ;;
            down)
                ((selected++))
                if [ $selected -ge $num_options ]; then
                    selected=0
                fi
                ;;
            enter)
                case $selected in
                    0) action_auto_install ;;
                    1) action_interactive_install ;;
                    2) action_skip_model_install ;;
                    3) action_manage ;;
                    4) action_update ;;
                    5) action_open_webui ;;
                    6) action_quick_start ;;
                    7) action_uninstall ;;
                    8)
                        clear
                        echo -e "${BLUE}[ℹ]${NC} Auf Wiedersehen!"
                        echo ""
                        exit 0
                        ;;
                esac
                ;;
        esac
    done
}

# ============================================================================
# DISCLAIMER
# ============================================================================

show_disclaimer() {
    clear
    echo -e "${RED}${BOLD}"
    echo "╔════════════════════════════════════════════════════════════════╗"
    echo "║              WICHTIGER HAFTUNGSAUSSCHLUSS                      ║"
    echo "╚════════════════════════════════════════════════════════════════╝"
    echo -e "${NC}\n"

    echo -e "${YELLOW}Diese Software wird OHNE JEGLICHE GARANTIE bereitgestellt.${NC}"
    echo -e "${YELLOW}Die Nutzung erfolgt AUF EIGENE GEFAHR.${NC}\n"

    echo "Der Autor übernimmt KEINERLEI Haftung für:"
    echo "  • Datenverlust oder Beschädigung"
    echo "  • Systemabstürze oder Instabilitäten"
    echo "  • Konfigurationsänderungen"
    echo "  • Jegliche direkten oder indirekten Schäden"
    echo ""

    echo -e "${RED}${BOLD}WICHTIG: Erstellen Sie ein Backup vor der Installation!${NC}\n"

    echo -e "${BLUE}Vollständiger Disclaimer: LICENSE_DISCLAIMER.txt${NC}"
    echo ""
    echo -e "${BOLD}Durch Fortfahren akzeptieren Sie:${NC}"
    echo "  ✓ Nutzung auf eigene Gefahr"
    echo "  ✓ Keine Haftung des Autors"
    echo "  ✓ Verantwortung für System-Backups"
    echo ""

    read -p "Verstanden und akzeptieren? (ja/nein): " -r
    echo ""

    if [[ $REPLY =~ ^[JjYy]([Aa]|[Ee][Ss])?$ ]]; then
        echo -e "${GREEN}✓ Bedingungen akzeptiert${NC}"
        sleep 1
        return 0
    else
        echo -e "${RED}Bedingungen nicht akzeptiert. Programm wird beendet.${NC}"
        exit 0
    fi
}

# ============================================================================
# START
# ============================================================================

# Prüfe ob im richtigen Verzeichnis
if [[ ! -f "./install.sh" ]] || [[ ! -f "./uninstall.sh" ]]; then
    echo -e "${RED}[✗]${NC} Fehler: Bitte führe dieses Script im OllamaDeploy-Verzeichnis aus"
    exit 1
fi

# Zeige Disclaimer beim ersten Start
show_disclaimer

# Starte Hauptmenü
main_menu
