#!/bin/bash

###############################################################################
# OllamaDeploy - Deinstallation
# Entfernt alle installierten Komponenten
#
# Nutzung:
#   ./uninstall.sh              # Interaktiv mit Rückfragen
#   ./uninstall.sh --force      # Alles löschen ohne Rückfragen
#   ./uninstall.sh --keep-data  # Software entfernen, Daten behalten
###############################################################################

set -e

# ============================================================================
# KONFIGURATION
# ============================================================================

VERSION="1.0.0"
FORCE_MODE=false
KEEP_DATA=false
KEEP_HOMEBREW=true

# ============================================================================
# FARBEN
# ============================================================================

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
BOLD='\033[1m'
NC='\033[0m'

# ============================================================================
# LOGGING
# ============================================================================

log_info() {
    echo -e "${BLUE}[ℹ]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[✓]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[⚠]${NC} $1"
}

log_error() {
    echo -e "${RED}[✗]${NC} $1"
}

log_step() {
    echo -e "\n${BLUE}${BOLD}→ $1${NC}"
}

# ============================================================================
# HILFSFUNKTIONEN
# ============================================================================

print_header() {
    clear
    echo -e "${RED}${BOLD}"
    echo "╔════════════════════════════════════════════════════════════════╗"
    echo "║                                                                ║"
    echo "║           OllamaDeploy Deinstallation v${VERSION}               ║"
    echo "║                                                                ║"
    echo "╚════════════════════════════════════════════════════════════════╝"
    echo -e "${NC}\n"
}

show_help() {
    cat << EOF
${BOLD}OllamaDeploy Deinstallation v${VERSION}${NC}

Entfernt alle installierten Komponenten von OllamaDeploy.

${BOLD}VERWENDUNG:${NC}
    ./uninstall.sh [OPTIONEN]

${BOLD}OPTIONEN:${NC}
    --force             Alles löschen ohne Rückfragen
    --keep-data         Software entfernen, Daten behalten
    --remove-homebrew   Homebrew auch entfernen (sonst behalten)
    --help              Diese Hilfe anzeigen

${BOLD}BEISPIELE:${NC}
    ./uninstall.sh                    # Interaktiv
    ./uninstall.sh --force            # Alles löschen
    ./uninstall.sh --keep-data        # Software weg, Daten behalten

${BOLD}WAS WIRD ENTFERNT:${NC}
    • Open WebUI Container
    • Ollama + alle Modelle
    • Docker Desktop
    • Optional: Homebrew
    • Optional: Alle Daten

${BOLD}DATEN-SPEICHERORTE:${NC}
    • Ollama-Modelle: ~/.ollama
    • Open WebUI: Docker Volume 'open-webui'
    • Docker-Daten: ~/Library/Containers/com.docker.docker
EOF
    exit 0
}

parse_arguments() {
    while [[ $# -gt 0 ]]; do
        case $1 in
            --force)
                FORCE_MODE=true
                shift
                ;;
            --keep-data)
                KEEP_DATA=true
                shift
                ;;
            --remove-homebrew)
                KEEP_HOMEBREW=false
                shift
                ;;
            --help|-h)
                show_help
                ;;
            *)
                log_error "Unbekannte Option: $1"
                echo "Nutze --help für weitere Informationen"
                exit 1
                ;;
        esac
    done
}

ask_confirmation() {
    if [[ "$FORCE_MODE" == "true" ]]; then
        return 0
    fi

    local prompt="$1"
    read -p "$prompt [y/N] " -n 1 -r
    echo

    [[ $REPLY =~ ^[Yy]$ ]]
}

# ============================================================================
# DEINSTALLATION
# ============================================================================

remove_openwebui() {
    log_step "Entferne Open WebUI..."

    if ! command -v docker &> /dev/null; then
        log_info "Docker nicht installiert, überspringe"
        return 0
    fi

    # Stoppe und entferne Container
    if docker ps -a --format '{{.Names}}' | grep -q "^open-webui$"; then
        log_info "Stoppe Container..."
        docker stop open-webui &> /dev/null || true

        log_info "Entferne Container..."
        docker rm open-webui &> /dev/null || true

        log_success "Container entfernt"
    else
        log_info "Container nicht gefunden"
    fi

    # Entferne Volume (wenn nicht --keep-data)
    if [[ "$KEEP_DATA" != "true" ]]; then
        if docker volume ls | grep -q "open-webui"; then
            if ask_confirmation "Open WebUI Daten löschen?"; then
                docker volume rm open-webui &> /dev/null || true
                log_success "Daten gelöscht"
            else
                log_info "Daten behalten"
            fi
        fi
    else
        log_info "Daten werden behalten (--keep-data)"
    fi

    # Entferne Image
    if docker images | grep -q "open-webui"; then
        docker rmi ghcr.io/open-webui/open-webui:main &> /dev/null || true
    fi
}

remove_ollama() {
    log_step "Entferne Ollama..."

    if ! command -v ollama &> /dev/null; then
        log_info "Ollama nicht installiert, überspringe"
        return 0
    fi

    # Stoppe Service
    if command -v brew &> /dev/null; then
        brew services stop ollama &> /dev/null || true
    fi

    # Entferne Ollama via Homebrew
    if command -v brew &> /dev/null; then
        if brew list | grep -q "^ollama$"; then
            log_info "Deinstalliere Ollama..."
            brew uninstall ollama &> /dev/null || true
            log_success "Ollama deinstalliert"
        fi
    fi

    # Entferne Daten (wenn nicht --keep-data)
    if [[ "$KEEP_DATA" != "true" ]] && [[ -d ~/.ollama ]]; then
        local ollama_size=$(du -sh ~/.ollama 2>/dev/null | cut -f1)

        if ask_confirmation "Ollama-Modelle löschen? (${ollama_size})"; then
            rm -rf ~/.ollama
            log_success "Modelle gelöscht"
        else
            log_info "Modelle behalten"
        fi
    else
        log_info "Daten werden behalten"
    fi
}

remove_docker() {
    log_step "Entferne Docker Desktop..."

    if [[ ! -d "/Applications/Docker.app" ]]; then
        log_info "Docker Desktop nicht installiert, überspringe"
        return 0
    fi

    # Docker beenden
    if pgrep -f "Docker.app" &> /dev/null; then
        log_info "Beende Docker Desktop..."
        osascript -e 'quit app "Docker"' &> /dev/null || true
        sleep 3
    fi

    # App entfernen
    if ask_confirmation "Docker Desktop deinstallieren?"; then
        rm -rf /Applications/Docker.app
        log_success "Docker Desktop entfernt"
    else
        log_info "Docker Desktop behalten"
        return 0
    fi

    # Daten entfernen (wenn nicht --keep-data)
    if [[ "$KEEP_DATA" != "true" ]]; then
        local docker_data=~/Library/Containers/com.docker.docker

        if [[ -d "$docker_data" ]]; then
            local docker_size=$(du -sh "$docker_data" 2>/dev/null | cut -f1)

            if ask_confirmation "Docker-Daten löschen? (${docker_size})"; then
                rm -rf "$docker_data"
                rm -rf ~/Library/Group\ Containers/group.com.docker
                rm -rf ~/.docker
                log_success "Docker-Daten gelöscht"
            else
                log_info "Docker-Daten behalten"
            fi
        fi
    else
        log_info "Daten werden behalten"
    fi
}

remove_homebrew() {
    if [[ "$KEEP_HOMEBREW" == "true" ]]; then
        log_info "Homebrew wird behalten (Standard)"
        return 0
    fi

    log_step "Entferne Homebrew..."

    if ! command -v brew &> /dev/null; then
        log_info "Homebrew nicht installiert, überspringe"
        return 0
    fi

    log_warning "Homebrew wird von vielen anderen Programmen verwendet!"

    if ask_confirmation "Homebrew wirklich deinstallieren?"; then
        /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/uninstall.sh)"
        log_success "Homebrew deinstalliert"
    else
        log_info "Homebrew behalten"
    fi
}

cleanup_system() {
    log_step "Räume System auf..."

    # Temporäre Dateien
    if [[ -d /tmp/ollamadeploy-install ]]; then
        rm -rf /tmp/ollamadeploy-install
    fi

    log_success "System aufgeräumt"
}

# ============================================================================
# ZUSAMMENFASSUNG
# ============================================================================

show_summary() {
    echo ""
    echo -e "${YELLOW}${BOLD}═══ Was wird entfernt? ═══${NC}\n"

    local items=()

    if docker ps -a --format '{{.Names}}' | grep -q "^open-webui$"; then
        items+=("✓ Open WebUI Container")
    fi

    if command -v ollama &> /dev/null; then
        items+=("✓ Ollama")
    fi

    if [[ -d "/Applications/Docker.app" ]]; then
        items+=("✓ Docker Desktop")
    fi

    if [[ "$KEEP_HOMEBREW" != "true" ]] && command -v brew &> /dev/null; then
        items+=("✓ Homebrew")
    fi

    if [[ "$KEEP_DATA" != "true" ]]; then
        items+=("✓ Alle gespeicherten Daten")
    else
        items+=("✗ Daten werden behalten")
    fi

    for item in "${items[@]}"; do
        echo "  $item"
    done

    echo ""

    if [[ ${#items[@]} -eq 0 ]]; then
        log_info "Nichts zu entfernen - bereits sauber!"
        exit 0
    fi
}

show_completion() {
    echo ""
    echo -e "${GREEN}${BOLD}"
    echo "╔════════════════════════════════════════════════════════════════╗"
    echo "║                                                                ║"
    echo "║            Deinstallation erfolgreich abgeschlossen!           ║"
    echo "║                                                                ║"
    echo "╚════════════════════════════════════════════════════════════════╝"
    echo -e "${NC}\n"

    log_success "Alle Komponenten entfernt"

    if [[ "$KEEP_DATA" == "true" ]]; then
        echo ""
        log_info "Deine Daten wurden behalten in:"
        echo "  • ~/.ollama (Modelle)"
        echo "  • Docker Volume 'open-webui' (Chats)"
        echo ""
        log_info "Bei Neuinstallation werden diese wiederverwendet"
    fi

    echo ""
    log_info "Zum erneuten Installieren: ./install.sh"
    echo ""
}

# ============================================================================
# MAIN
# ============================================================================

main() {
    parse_arguments "$@"

    print_header

    if [[ "$FORCE_MODE" == "true" ]]; then
        log_warning "Force-Modus aktiviert - keine Rückfragen!"
        echo ""
    fi

    show_summary

    # Finale Bestätigung
    if [[ "$FORCE_MODE" != "true" ]]; then
        echo -e "${RED}${BOLD}WARNUNG:${NC} Diese Aktion kann nicht rückgängig gemacht werden!\n"

        if ! ask_confirmation "Wirklich deinstallieren?"; then
            log_info "Deinstallation abgebrochen"
            exit 0
        fi

        echo ""
    fi

    # Deinstallation
    remove_openwebui
    remove_ollama
    remove_docker
    remove_homebrew
    cleanup_system

    # Abschluss
    show_completion
}

# Start
main "$@"
